<?php
// lib/RateLimiter.php
class RateLimiter {
    private $db;
    private $perMinute;

    public function __construct($mysqli, $perMinute = 60) {
        $this->db = $mysqli;
        $this->perMinute = $perMinute;
    }

    public function checkAndIncrement($apiKeyId, $clientIp, $dailyLimit) {
        $now = new DateTime('now');
        $minuteWindow = (clone $now)->setTime((int)$now->format('H'), (int)$now->format('i'), 0);
        $dayStart = (clone $now)->setTime(0, 0, 0);

        $minuteWindowStr = $minuteWindow->format('Y-m-d H:i:s');
        $dayStartStr = $dayStart->format('Y-m-d H:i:s');

        // 1) Per-minute limit for this key + client IP
        $stmt = $this->db->prepare("
            SELECT id, request_count FROM rate_limits
             WHERE api_key_id = ? AND ip = ? AND window_type = 'minute' AND window_start = ?
        ");
        $stmt->bind_param('iss', $apiKeyId, $clientIp, $minuteWindowStr);
        $stmt->execute();
        $stmt->bind_result($id, $count);
        if ($stmt->fetch()) {
            $stmt->close();
            if ($count >= $this->perMinute) {
                return ['ok' => false, 'reason' => 'minute_limit_reached'];
            }
            $stmt = $this->db->prepare("UPDATE rate_limits SET request_count = request_count + 1 WHERE id = ?");
            $stmt->bind_param('i', $id);
            $stmt->execute();
            $stmt->close();
        } else {
            $stmt->close();
            $stmt = $this->db->prepare("
                INSERT INTO rate_limits (api_key_id, ip, window_start, window_type, request_count)
                VALUES (?, ?, ?, 'minute', 1)
            ");
            $stmt->bind_param('iss', $apiKeyId, $clientIp, $minuteWindowStr);
            $stmt->execute();
            $stmt->close();
        }

        // 2) Daily limit (per api_key, ignore IP)
        $stmt = $this->db->prepare("
            SELECT request_count FROM rate_limits
            WHERE api_key_id = ? AND window_type = 'day' AND window_start = ?
            LIMIT 1
        ");
        $stmt->bind_param('is', $apiKeyId, $dayStartStr);
        $stmt->execute();
        $stmt->bind_result($dayCount);
        if ($stmt->fetch()) {
            $todayCount = (int)$dayCount;
            $stmt->close();
        } else {
            $stmt->close();
            $todayCount = 0;
        }

        if ($todayCount >= $dailyLimit) {
            return ['ok' => false, 'reason' => 'daily_limit_reached'];
        }

        // Increment or create daily window
        if ($todayCount === 0) {
            $stmt = $this->db->prepare("
                INSERT INTO rate_limits (api_key_id, ip, window_start, window_type, request_count)
                VALUES (?, '', ?, 'day', 1)
            ");
            $stmt->bind_param('is', $apiKeyId, $dayStartStr);
        } else {
            $stmt = $this->db->prepare("
                UPDATE rate_limits
                SET request_count = request_count + 1
                WHERE api_key_id = ? AND window_type = 'day' AND window_start = ?
            ");
            $stmt->bind_param('is', $apiKeyId, $dayStartStr);
        }
        $stmt->execute();
        $stmt->close();

        return ['ok' => true];
    }
}
?>
