<?php
// lib/GeoIP.php
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../geo/ip2location.php'; // include IP2Location PHP library

class GeoIP {
    private $db;
    private $ip2location;

    public function __construct($mysqli) {
        $this->db = $mysqli;
        $this->ip2location = new \IP2Location\Database(
            __DIR__ . '/../geo/IP2LOCATION-LITE-DB3.IPV6.BIN',
            \IP2Location\Database::FILE_IO
        );
    }

    public function lookup($ip) {
        // 1) Check cache first
        $stmt = $this->db->prepare("SELECT country, region, city, latitude, longitude, timezone FROM ip_cache WHERE ip = ?");
        $stmt->bind_param('s', $ip);
        $stmt->execute();
        $stmt->bind_result($country, $region, $city, $lat, $lon, $tz);
        if ($stmt->fetch()) {
            $stmt->close();
            return [
                'ip'        => $ip,
                'country'   => $country,
                'region'    => $region,
                'city'      => $city,
                'latitude'  => $lat,
                'longitude' => $lon,
                'timezone'  => $tz,
                'source'    => 'cache'
            ];
        }
        $stmt->close();

        // 2) Query IP2Location DB
        $record = $this->ip2location->lookup($ip, \IP2Location\Database::ALL);

        $country = $record['countryName'] ?? null;
        $region  = $record['regionName'] ?? null;
        $city    = $record['cityName'] ?? null;
        $lat     = $record['latitude'] ?? null;
        $lon     = $record['longitude'] ?? null;
        $tz      = $record['timeZone'] ?? null;

        // 3) Save to cache (upsert)
        $stmt = $this->db->prepare("
            INSERT INTO ip_cache (ip, country, region, city, latitude, longitude, timezone, last_updated)
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ON DUPLICATE KEY UPDATE
              country = VALUES(country),
              region = VALUES(region),
              city = VALUES(city),
              latitude = VALUES(latitude),
              longitude = VALUES(longitude),
              timezone = VALUES(timezone),
              last_updated = NOW()
        ");
        $stmt->bind_param('ssssdds', $ip, $country, $region, $city, $lat, $lon, $tz);
        $stmt->execute();
        $stmt->close();

        return [
            'ip'        => $ip,
            'country'   => $country,
            'region'    => $region,
            'city'      => $city,
            'latitude'  => $lat,
            'longitude' => $lon,
            'timezone'  => $tz,
            'source'    => 'db'
        ];
    }
}
?>
